//Firmware do kalibratora DS3231 z serwerem NTP
//baza: NodeMCU ESP8266

//Arduino board manager: esp8266 3.1.2
//Arduino board: NodeMCU 1.0 (ESP-12E Module)
//Adafruit GFX Library 1.12.4
//Adafruit SSD1306 2.2.3

//obsluga kart SD
//SCK  - GPIO 14 - D5
//MISO - GPIO 12 - D6
//MOSI - GPIO 13 - D7
//CS   - GPIO 15 - D8

//#define DEBUG 1
#if DEBUG
#define DBG(x) Serial.println(x)
#else
#define DBG(x)  //nic
#endif

#include <SPI.h>
#include <SD.h>
#include <Wire.h>
#include <EEPROM.h>
#include <ESP8266WiFi.h>
#include <WiFiUdp.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>

//konfiguracja pinow i innych rzeczy
#define PIN_CS D8
#define PIN_SDA D4
#define PIN_SCL D3
#define PIN_SQW D1
#define PIN_BUTTON_UP D0
#define PIN_BUTTON_DOWN D2

#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET -1  // brak pinu RESET
#define OLED_ADDR 0x3C

#define DS3231_ADDR 0x68
#define DS3231_CONTROL_REG 0x0E
#define DS3231_STATUS_REG 0x0F
#define DS3231_OFFSET_REG 0x10

#define WIFI_MAX_SSID 32
#define WIFI_MAX_PASS 64
#define WIFI_MAX_NTPDOMAIN 64
#define EEPROM_SIZE (WIFI_MAX_SSID + WIFI_MAX_PASS + WIFI_MAX_NTPDOMAIN)
#define EEPROM_SSID_ADDR 0
#define EEPROM_PASS_ADDR (EEPROM_SSID_ADDR + WIFI_MAX_SSID)
#define EEPROM_NTPDOMAIN_ADDR (EEPROM_PASS_ADDR + WIFI_MAX_PASS)

#define SCREENSAVER_TIMER 3000  //5 minut

//zmienne globalne
String wifi_ssid;
String wifi_pass;
String wifi_ntpdomain;
uint64_t cnt_ntp_begin, cnt_ntp_end;        //liczniki czasu pierwszego i ostatniego pomiaru ntp z dokladnoscia 1ms.
volatile uint32_t cnt_ds_1s;                //licznik impulsow z ds3231 co 1 sekunde
volatile uint32_t cnt_ds_micros;            //stan sprzetowego licznika us, w momencie impulsu z DS3231
uint16_t cnt_ds_1ms_begin, cnt_ds_1ms_end;  //dodatkowa ilosc ms od impulsu DS pierwszego i ostatniego pomiaru
int16_t ds_ppm;                             //policzony ppm
int8_t ds_offset;                           //wyznaczony offset
uint8_t button_no = 0;                      //nr nacisnietego przycisku: 0-brak, 1-UP, 2-DOWN, 3-UP+DOWN
uint8_t button_sec = 0;                     //czas trzymania przycisku w sekundach
uint16_t accu_voltage = 0;                  //napiecie akumulatora fixed point 0.00
uint16_t screen_timer = SCREENSAVER_TIMER;  //licznik wygaszacza ekranu
int8_t ds_old_offset = 0;                   //odczytany z ds3231 offset

Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

///////////////////////////////////////////////
/////////////// PRZERWANIE

void IRAM_ATTR onSQW() {
  cnt_ds_1s++;
  cnt_ds_micros = micros();
}

///////////////////////////////////////////////
/////////////// OBSLUGA RTC

//odczyt offsetu z DS3231
void DS3231_GetOffset() {
  DBG("Odczyt offset z I2C DS2131");
  Wire.beginTransmission(DS3231_ADDR);
  Wire.write(DS3231_OFFSET_REG);
  Wire.endTransmission(false);
  Wire.requestFrom(DS3231_ADDR, 1);
  if (Wire.available()) {
    ds_old_offset = (int8_t)Wire.read();
    DBG("Odczytano offset z I2C DS2131: " + ds_old_offset);
  }
}

//ustawienie offsetu w DS3231
void DS3231_SetOffset(int8_t offset) {
  Wire.beginTransmission(DS3231_ADDR);
  Wire.write(DS3231_OFFSET_REG);
  Wire.write((uint8_t)offset);
  uint8_t err = Wire.endTransmission();
  if (err) {
    DBG("Blad I2C przy zapisie offset DS2131:" + err);
    return;
  }
  DBG("Zapisano offset DS3231 przez I2C");
}

//ustawienie DS3231
void DS3231_Setup() {
  Wire.beginTransmission(DS3231_ADDR);
  Wire.write(DS3231_CONTROL_REG);
  Wire.write(0b00000000);  //wlacz osc, brak sqw przy baterii, nie mierz teraz temp, 1Hz SQW, wlacz SQW zamiast INT, wylacz oba alarmy
  uint8_t err = Wire.endTransmission();
  if (err) {
    DBG("Blad I2C przy DS2131:" + err);
    return;
  }
  DBG("Skonfigurowano DS3231 przez I2C");
  DS3231_GetOffset();
}

//przestawianie offset
void DS3231_ChangeOffset(int8_t offset) {
  if (offset == 0) return;  //nie ma sensu zmieniac o wartosc 0
  int16_t new_offset = (int16_t)ds_old_offset + (int8_t)offset;
  if (new_offset > 127) new_offset = 127;
  if (new_offset < -128) new_offset = -128;
  DS3231_SetOffset((int8_t)new_offset);
  DS3231_GetOffset();  //ponowny odczyt / weryfikacja
}

///////////////////////////////////////////////
/////////////// OBSLUGA WIFI I NTP

//polaczenie z Wifi i odczyt czasu z ntp (0=blad, inna liczba to liczba ms od roku 1900)
uint64_t GetNTPTime() {
  WiFiUDP udp;
  const unsigned int localPort = 2390;
  const int NTP_PACKET_SIZE = 48;
  IPAddress ntpAddr;
  byte packetBuffer[NTP_PACKET_SIZE];

  DBG("Laczenie z WiFi");
  WiFi.forceSleepWake();
  delay(1);
  WiFi.mode(WIFI_STA);  //tryb stacji
  WiFi.begin(wifi_ssid, wifi_pass);
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    DBG(".");
  }
  udp.begin(localPort);

  DBG("Zmiana domeny na IP");
  WiFi.hostByName(wifi_ntpdomain.c_str(), ntpAddr);

  DBG("Laczenie z NTP");
  uint32_t timer_start = micros();

  memset(packetBuffer, 0, NTP_PACKET_SIZE);
  packetBuffer[0] = 0b11100011;   // LI, Version, Mode (client)
  udp.beginPacket(ntpAddr, 123);  // port NTP
  udp.write(packetBuffer, NTP_PACKET_SIZE);
  udp.endPacket();

  bool sukces = true;
  uint32_t startWait = millis();
  while (!udp.parsePacket()) {
    if (millis() - startWait > 2000) {  // timeout 2s
      sukces = false;
      break;
    }
  }

  uint64_t wynik = 0;
  if (sukces) {
    uint32_t timer_delta = micros() - timer_start;  //zmierzony czas laczenia z ntp w us
    uint16_t ntp_delay = timer_delta / 2000;        //czas przeliczony na ms i wzieta polowa (odczyt z sieci)
    udp.read(packetBuffer, NTP_PACKET_SIZE);
    // 4 bajty – sekundy od 1900
    uint32_t secsSince1900 = (packetBuffer[40] << 24) | (packetBuffer[41] << 16) | (packetBuffer[42] << 8) | packetBuffer[43];
    // 4 bajty – ułamek sekundy
    uint32_t frac = (packetBuffer[44] << 24) | (packetBuffer[45] << 16) | (packetBuffer[46] << 8) | packetBuffer[47];
    // przelicz frakcję na milisekundy
    uint32_t ms = (uint64_t)frac * 1000 >> 32;
    wynik = secsSince1900 * 1000 + ms - ntp_delay;
    DBG(String("Czas NTP: ") + secsSince1900 + "." + ms + "  czas transmisji: " + ntp_delay);
  } else {
    DBG("Brak odpowiedzi NTP");
  }
  udp.stop();
  WiFi.disconnect(true);
  WiFi.mode(WIFI_OFF);
  WiFi.forceSleepBegin();
  return wynik;
}

///////////////////////////////////////////////
/////////////// POMIARY

void PomiarAku() {
  uint16_t adc = analogRead(A0);
  //Ubat=4,58156*ADC+148  0.000
  //Ubat=0,45816*ADC+15  0.00
  accu_voltage = (uint16_t)((uint32_t)adc * 45816 / 100000) + 15;
  //  accu_voltage = adc;
}

void PomiarRozpocznij() {
  uint64_t ntp_time = GetNTPTime();
  if (ntp_time != 0) cnt_ntp_begin = ntp_time;
  else {
    cnt_ntp_begin = 0;
    //XXXX miejsce na wyswietlenie komunikatu
  }
  //JAK NAJSZYBSZE zerowanie licznika impulsow z DS3231 po pomiarze NTP
  noInterrupts();
  cnt_ds_1s = 0;
  uint32_t cnt_delta_us = micros() - cnt_ds_micros;  //nawet jak sie licznik przekreci, to odejmowanie na uint da prawidlowy wynik
  interrupts();
  //KONIEC SEKCJI JAK NAJSZYBSZEJ
  cnt_delta_us /= 1000;                        //przeliczenie na ms
  if (cnt_delta_us > 999) cnt_delta_us = 999;  //gdy brak impulsow z dsa to przyjmujemy max 999ms
  cnt_ds_1ms_begin = cnt_delta_us;             //zapisz ms poczatku pomiaru
  //na poczatku pomiarow ustawiamy liczniki koncowe i wyniki obliczen
  cnt_ntp_end = cnt_ntp_begin;
  cnt_ds_1ms_end = cnt_ds_1ms_begin;
  ds_ppm = 0;     //wstepnie blad ustaw na 0
  ds_offset = 0;  //wstepnie offset ustaw na 0
}

void PomiarKontynuuj() {
  uint64_t ntp_time = GetNTPTime();
  if (ntp_time != 0) cnt_ntp_end = ntp_time;
  else {
    //XXXX miejsce na wyswietlenie komunikatu
    return;  //nic dalej nie robimy, niech uzytkownik jeszcze raz robi pomiar
  }
  //JAK NAJSZYBSZY odczyt mikrosekund od ostatniego impulsu z DS3231 po pomiarze NTP
  noInterrupts();
  uint32_t cnt_delta_us = micros() - cnt_ds_micros;  //nawet jak sie licznik przekreci, to odejmowanie na uint da prawidlowy wynik
  interrupts();
  //KONIEC SEKCJI JAK NAJSZYBSZEJ
  cnt_delta_us /= 1000;                        //przeliczenie na ms
  if (cnt_delta_us > 999) cnt_delta_us = 999;  //gdy brak impulsow z dsa to przyjmujemy max 999ms
  cnt_ds_1ms_end = cnt_delta_us;               //zapisz ms pomiaru
  //w czasie kontynuacji obliczamy blad i offset
  //jeżeli nie będziemy dzielić przez 0
  ntp_time = cnt_ntp_end - cnt_ntp_begin;
  if (ntp_time == 0) {
    ds_ppm = 0;     //wstepnie blad ustaw na 0
    ds_offset = 0;  //wstepnie offset ustaw na 0
    return;
  }
  //wartosc wzgledna bledu w ppm: (czas_ds - czas_ntp)/czas_ntp * 1 000 000
  uint64_t ds_time = cnt_ds_1s * 1000 + cnt_ds_1ms_end - cnt_ds_1ms_begin;
  //wynik bledu dla dokladnosci x100
  int64_t blad = 100000000 * ((int64_t)ds_time - (int64_t)ntp_time) / (int64_t)ntp_time;
  //bezpiecznik dla kosmicznie duzych, czy malych bledow PPM
  if (blad > 30000) blad = 30000;
  if (blad < -30000) blad = -30000;
  ds_ppm = int16_t(blad);
  //przy 25 stC na jeden bit offsetu przypada korekta około 0.07ppm (zwiększanie offsetu spowalnia zegar)
  //ofs=ppm/100/0.07=ppm/100/(7/100)=ppm/7
  int16_t ofs = ds_ppm / 7;  //przerabiamy na offset uwzględniając że ppm było mnożone przez 100
  if (ofs > 127) ofs = 127;
  if (ofs < -128) ofs = -128;
  ds_offset = (int8_t)ofs;
}

///////////////////////////////////////////////
/////////////// WYSWIETLACZ
void WyswietlNapis(const char t[]) {
  display.clearDisplay();
  display.setTextColor(SSD1306_WHITE);
  display.setCursor(0, 0);
  display.setTextSize(2);
  display.printf("%s", t);
  display.display();
}

void WyswietlPomiary() {
  uint64_t disp_ntp, disp_ds;  //roznica czasu  w ms
  uint32_t disp_ntp_sec, disp_ds_sec;
  uint16_t disp_ntp_ms, disp_ds_ms;
  uint16_t u16;
  char t[50], napis[50];
  //pomiar akumulatora robiony tylko przy okazji wyswietlania pomiarow
  PomiarAku();
  //odstep czasu ntp (licznik sekund i ms)
  disp_ntp = cnt_ntp_end - cnt_ntp_begin;
  disp_ntp_sec = disp_ntp / 1000;
  disp_ntp_ms = disp_ntp % 1000;
  //odczep czasu rtc (licznik sekund i ms)
  disp_ds = cnt_ds_1s * 1000 + cnt_ds_1ms_end - cnt_ds_1ms_begin;
  disp_ds_sec = disp_ds / 1000;
  disp_ds_ms = disp_ds % 1000;
  //wyswietlanie
  DBG(String("NTP:") + disp_ntp_sec + "." + disp_ntp_ms + "  DS:" + disp_ds_sec + "." + disp_ds_ms + "  ErrPPM:" + ds_ppm + "  Offset:" + ds_offset);
  display.clearDisplay();
  display.setTextColor(SSD1306_WHITE);
  display.setCursor(0, 0);
  //wyswietlenie napiecia aku
  display.setTextSize(1);
  display.printf("Ubat: %u.%02uV\n", accu_voltage / 100, accu_voltage % 100);
  //wyswietlenie aktualnego offset odczytanego z DS
  display.printf("DS3231 Offset: %+04i\n", ds_old_offset);
  //wyswietlenie czasu mierzonego przez DS
  display.printf(" DS: %lu.%03u\n", disp_ds_sec, disp_ds_ms);
  //wyswietlenie czasu mierzonego z ntp
  display.printf("NTP: %lu.%03u\n", disp_ntp_sec, disp_ntp_ms);
  //wyswietlenie bledu ppm
  display.setTextSize(2);
  display.printf("PPM%+7.2f\n", ds_ppm / 100.0);
  //wyswietlenie proponowanej zmiany offsetu
  display.printf("Offset%+04i", ds_offset);
  display.display();
}

///////////////////////////////////////////////
/////////////// PRZYCISKI

//zwraca true jezeli cos jest nacisniete
bool ButtonIsPressed() {
  return !(digitalRead(PIN_BUTTON_UP) && digitalRead(PIN_BUTTON_DOWN));
}

//odczyt nr przycisku
void ButtonRead() {
  button_no = 0;
  button_sec = 0;
  if (!digitalRead(PIN_BUTTON_UP)) button_no = 1;
  if (!digitalRead(PIN_BUTTON_DOWN)) button_no += 2;  //wynikiem bedzie 2 lub 3 zaleznie od tego czy UP jest nacisiniety
  if (button_no == 0) return;                         //nic nie robimy gdy nic nie nacisnieto
  uint32_t usec = micros();
  while (ButtonIsPressed())
    delay(10);  //czekamy w petli na puszczenie przycisku
  usec = (micros() - usec) / 1000000;
  button_sec = (uint8_t)usec;  //czas trzymania w sekundach
}

//przycisk wyjscia z wygaszacza
void ButtonUpShort() {
  DBG("Przycisk wyjscia z wygaszacza");
  WyswietlPomiary();
}

//przycisk zapisu_offsetu
void ButtonUpLong() {
  DBG("Przycisk zapisz offset");
  WyswietlNapis("Zapisuje\noffset...");
  DS3231_ChangeOffset(ds_offset);
  WyswietlPomiary();
}

//przycisk aktualizacji pomiarow
void ButtonDownShort() {
  DBG("Przycisk kontynuacja");
  WyswietlNapis("Kolejny\npomiar...");
  PomiarKontynuuj();
  WyswietlPomiary();
}

//przycisk poczatku pomiarow
void ButtonDownLong() {
  DBG("Przycisk poczatek pomiarow");
  WyswietlNapis("Pierwszy\npomiar...");
  PomiarRozpocznij();
  WyswietlPomiary();
}

///////////////////////////////////////////////
/////////////// SETUP I LOOP

void setup() {
  File configfile;
#ifdef DEBUG
  Serial.begin(9600);
  Serial.println("");
#endif
  pinMode(PIN_BUTTON_UP, INPUT_PULLUP);
  pinMode(PIN_BUTTON_DOWN, INPUT_PULLUP);
  Wire.begin(PIN_SDA, PIN_SCL);  //wlacz i2c
  //aktywacja wyswietlacza
  if (!display.begin(SSD1306_SWITCHCAPVCC, OLED_ADDR)) {
    DBG("OLED init error");
    while (1)
      ;  //nie ma sensu kontynuowac skoro nic nie widac
  }
  display.clearDisplay();
  display.display();
  //przygotowanie mierzenia impulsow z DS3231
  pinMode(PIN_SQW, INPUT);
  DS3231_Setup();            //ustawienie SQW w DS3231
  cnt_ds_1s = 0;             //wstepne wyzerowanie licznika
  cnt_ds_micros = micros();  //wstepne ustawienie licznika mikrosekund impulsow z DS
  attachInterrupt(digitalPinToInterrupt(PIN_SQW), onSQW, RISING);
  //przygotowanie zmiennych do konfiguracji WIFI
  wifi_ssid.reserve(WIFI_MAX_SSID);
  wifi_pass.reserve(WIFI_MAX_PASS);
  wifi_ntpdomain.reserve(WIFI_MAX_NTPDOMAIN);
  EEPROM.begin(EEPROM_SIZE);
  //Odczyt konfiguracji Wifi z karty SD
  DBG("Odczyt konfiguracji Wifi z SD");
  if (SD.begin(PIN_CS)) {
    DBG("Prawidlowa inicjalizacja modulu SD");
    if (SD.exists("/wifi.txt")) {
      DBG("Wykryto plik konfiguracji Wifi");
      configfile = SD.open("/wifi.txt", FILE_READ);
      if (configfile) {
        String t;
        t.reserve(WIFI_MAX_PASS);
        configfile.setTimeout(500);
        //odczyt ssid (max 32 znaki)
        t = configfile.readStringUntil('\n');
        t.trim();
        wifi_ssid = t.substring(0, WIFI_MAX_SSID);
        DBG("SSID:" + wifi_ssid);
        //odczyt pass (max 64 znaki)
        t = configfile.readStringUntil('\n');
        t.trim();
        wifi_pass = t.substring(0, WIFI_MAX_PASS);
        DBG("Pass:" + wifi_pass);
        //odczyt domeny ntp (max 64 znaki)
        t = configfile.readStringUntil('\n');
        t.trim();
        wifi_ntpdomain = t.substring(0, WIFI_MAX_NTPDOMAIN);
        DBG("NTP domain:" + wifi_ntpdomain);
        configfile.close();
        //zapisanie wczytanych danych do EEPROM
        DBG("Zapis konfiguracji do EEPROM");
        for (int i = 0; i < WIFI_MAX_SSID; i++) {
          if (i < wifi_ssid.length()) EEPROM.write(EEPROM_SSID_ADDR + i, wifi_ssid[i]);
          else EEPROM.write(EEPROM_SSID_ADDR + i, 0);  // wypełnienie zerami
        }
        for (int i = 0; i < WIFI_MAX_PASS; i++) {
          if (i < wifi_pass.length()) EEPROM.write(EEPROM_PASS_ADDR + i, wifi_pass[i]);
          else EEPROM.write(EEPROM_PASS_ADDR + i, 0);  // wypełnienie zerami
        }
        for (int i = 0; i < WIFI_MAX_NTPDOMAIN; i++) {
          if (i < wifi_ntpdomain.length()) EEPROM.write(EEPROM_NTPDOMAIN_ADDR + i, wifi_ntpdomain[i]);
          else EEPROM.write(EEPROM_NTPDOMAIN_ADDR + i, 0);  // wypełnienie zerami
        }
        EEPROM.commit();
      }
    }
  }  //koniec przepisywania konfiguracji z SD do EEPROM
  DBG("Koniec konfiguracji z SD");
  //odczyt konfiguracji z EEPROM
  char buf[WIFI_MAX_PASS + 1];  //pass jest dluzsze niz ssid
  for (int i = 0; i < WIFI_MAX_SSID; i++) buf[i] = EEPROM.read(EEPROM_SSID_ADDR + i);
  buf[WIFI_MAX_SSID] = 0;
  wifi_ssid = buf;
  DBG("eeprom ssid:" + wifi_ssid);
  for (int i = 0; i < WIFI_MAX_PASS; i++) buf[i] = EEPROM.read(EEPROM_PASS_ADDR + i);
  buf[WIFI_MAX_PASS] = 0;
  wifi_pass = buf;
  DBG("eeprom pass:" + wifi_pass);
  for (int i = 0; i < WIFI_MAX_NTPDOMAIN; i++) buf[i] = EEPROM.read(EEPROM_NTPDOMAIN_ADDR + i);
  buf[WIFI_MAX_NTPDOMAIN] = 0;
  wifi_ntpdomain = buf;
  DBG("eeprom ntpdomain:" + wifi_ntpdomain);
  //wstepne wyzerowanie pomiarow po wlaczeniu zeby pokazywalo na wyswietlaczu cos sensownego
  WyswietlNapis("Wstepny\npomiar...");
  PomiarRozpocznij();
  WyswietlPomiary();
}

void loop() {
  //obsluga klawiatury
  if (ButtonIsPressed()) {
    screen_timer = SCREENSAVER_TIMER;  //po nacisnieciu klawisza zresetuj timer wygaszacza ekranu
    ButtonRead();
    if (button_no == 1) {
      if (button_sec < 3) ButtonUpShort();
      else ButtonUpLong();
    }
    if (button_no == 2) {
      if (button_sec < 3) ButtonDownShort();
      else ButtonDownLong();
    }
  }
  //wygaszacz ekranu
  if (screen_timer > 0) {
    screen_timer--;
    if (screen_timer == 0) {
      display.clearDisplay();
      display.display();
    }
  }
  delay(100);
}
